<?php

namespace App\Models\Lap;

use CodeIgniter\Model;
use CodeIgniter\I18n\Time;
use App\Models\PresensiInterface;
use App\Libraries\enums\Kehadiran;

class MPresensiSiswaModel extends Model implements PresensiInterface
{
    protected $table      = 'tb_presensi_siswa';
    protected $primaryKey = 'id_presensi';

    protected $allowedFields = [
        'id_siswa',
        'id_kelas',
        'tanggal',
        'jam_masuk',
        'jam_sholat',
        'jam_keluar',
        'id_kehadiran',
        'keterangan'
    ];

    /* =====================================================
     * INTERFACE METHODS
     * ===================================================== */

    public function cekAbsen(string|int $id, string|Time $date)
    {
        $row = $this->where([
            'id_siswa' => $id,
            'tanggal'  => $date
        ])->first();

        return $row ? $row[$this->primaryKey] : false;
    }

    public function absenMasuk(string $id, $date, $time)
    {
        $existing = $this->where([
            'id_siswa' => $id,
            'tanggal'  => $date
        ])->first();

        if ($existing) {
            return $this->update($existing[$this->primaryKey], [
                'jam_masuk'    => $time,
                'id_kehadiran' => Kehadiran::Hadir->value
            ]);
        }

        return $this->insert([
            'id_siswa'      => $id,
            'tanggal'       => $date,
            'jam_masuk'     => $time,
            'id_kehadiran'  => Kehadiran::Hadir->value,
            'keterangan'    => ''
        ]);
    }

    public function absenKeluar(string $idPresensi, $time)
    {
        return $this->update($idPresensi, [
            'jam_keluar' => $time
        ]);
    }

    public function getPresensiById(string $idPresensi)
    {
        return $this->where($this->primaryKey, $idPresensi)->first();
    }

    public function getPresensiByKehadiran(string $idKehadiran, $tanggal)
    {
        $builder = $this->db->table('tb_presensi_siswa p')
            ->select('p.*, s.nama_siswa, k.kelas AS nama_kelas')
            ->join('tb_siswa s', 's.id_siswa = p.id_siswa', 'left')
            ->join('tb_kelas k', 'k.id_kelas = p.id_kelas', 'left')
            ->where('p.tanggal', $tanggal);

        if ($idKehadiran === '4') {
            $builder->whereNotIn('p.id_kehadiran', ['1','2','3']);
        } else {
            $builder->where('p.id_kehadiran', $idKehadiran);
        }

        return $builder
            ->orderBy('s.nama_siswa', 'ASC')
            ->get()
            ->getResultArray();
    }

    /* =====================================================
     * METHODS KHUSUS LAPORAN
     * ===================================================== */

    public function getPresensiByTanggal($tanggal)
    {
        return $this->db->table('tb_presensi_siswa p')
            ->select('p.*, s.nama_siswa, k.kelas AS nama_kelas')
            ->join('tb_siswa s', 's.id_siswa = p.id_siswa', 'left')
            ->join('tb_kelas k', 'k.id_kelas = p.id_kelas', 'left')
            ->where('p.tanggal', $tanggal)
            ->orderBy('k.kelas', 'ASC')
            ->orderBy('s.nama_siswa', 'ASC')
            ->get()
            ->getResultArray();
    }

    public function getPresensiByKelasTanggal($idKelas, $tanggal)
    {
        return $this->db->table('tb_presensi_siswa p')
            ->select('p.*, s.nama_siswa, k.kelas AS nama_kelas')
            ->join('tb_siswa s', 's.id_siswa = p.id_siswa', 'left')
            ->join('tb_kelas k', 'k.id_kelas = p.id_kelas', 'left')
            ->where('p.id_kelas', $idKelas)
            ->where('p.tanggal', $tanggal)
            ->orderBy('s.nama_siswa', 'ASC')
            ->get()
            ->getResultArray();
    }

    public function getPresensiByTanggalRange($tanggalAwal, $tanggalAkhir, $idKelas = null)
    {
        $builder = $this->db->table('tb_presensi_siswa p')
            ->select('p.*, s.id_siswa, s.nama_siswa, s.jenis_kelamin, k.kelas AS nama_kelas')
            ->join('tb_siswa s', 's.id_siswa = p.id_siswa', 'left')
            ->join('tb_kelas k', 'k.id_kelas = s.id_kelas', 'left')
            ->where('p.tanggal >=', $tanggalAwal)
            ->where('p.tanggal <=', $tanggalAkhir);

        if ($idKelas && $idKelas !== 'all') {
            $builder->where('s.id_kelas', $idKelas);
        }

        $builder->orderBy('p.id_siswa, p.tanggal ASC');

        return $builder->get()->getResultArray();
    }
}
